using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using System.Collections;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.DAL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/13/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access component providing database read/write functionality for
	///	blood unit tests data.
	///</summary>

	#endregion
	public class BloodUnitTest
	{
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="642"> 
		///		<ExpectedInput>Blood unit identifier, blood test type identifier and division code</ExpectedInput>
		///		<ExpectedOutput>The data table containing blood unit test results for a given blood unit and test type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1897"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood unit test results for a given blood unit and test type
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit GUID</param>
		/// <param name="bloodTestTypeID">Blood test type ID</param>
		/// <param name="divisionCode">Division code</param>
		/// <returns>Data table containing blood unit test results</returns>
		public static DataTable GetBloodUnitTestResult(Guid bloodUnitGuid, int bloodTestTypeID, string divisionCode)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitTestResult.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetBloodUnitTestResult.bloodtesttypeid, System.Data.SqlDbType.Int),
					new SqlParameter(STOREDPROC.GetBloodUnitTestResult.divisioncode, System.Data.SqlDbType.Char)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = bloodTestTypeID;
			prms[2].Value = divisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResult.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1657"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1658"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		/// Gets blood unit special test data for a blood unit with a given GUID
		///</summary>
		///<param name="bloodUnitGuid">Blood unit GUID</param>
		///<returns>DataTable containing blood unit special test data</returns>
		public static DataTable GetBloodUnitSpecialTest(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetAntiGenListForBloodUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier)
				};

			prms[0].Value = bloodUnitGuid;

			return Common.StoredProcedure.GetData(STOREDPROC.GetAntiGenListForBloodUnit.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1659"> 
		///		<ExpectedInput>BloodUnitGuid, Division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1660"> 
		///		<ExpectedInput>Invalid BloodUnitguid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		/// Gets blood unit special test results data for a blood unit with a given GUID
		///</summary>
		///<param name="bloodUnitGuids">Comma delimited (no spaces) list of blood unit guids</param>
		///<param name="divisionCode">Division Code</param>
		///<returns>DataTable containing blood unit special test results data</returns>
		public static DataTable GetBloodUnitTestResults(string bloodUnitGuids, string divisionCode)
		{
			SqlParameter[] prms =
				{
					// CR2899: changed input parameter to accept a list of bloodunit guids
					new SqlParameter(STOREDPROC.GetBloodUnitTestResults.bloodunitguidlist, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitTestResults.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuids;
			prms[1].Value = divisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResults.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3766"> 
		///		<ExpectedInput>Valid blood unit GUID and division code</ExpectedInput>
		///		<ExpectedOutput>A data table with ABO/Rh Confirmation test data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3767"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood unit Abo/Rh Confirmation tests data
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit GUID</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns></returns>
		public static DataTable GetAboRHConfirmationTests(Guid bloodUnitGuid, string divisionCode)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetAboRHConfirmationTestsForUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetAboRHConfirmationTestsForUnit.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = divisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetAboRHConfirmationTestsForUnit.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3768"> 
		///		<ExpectedInput>Valid blood unit GUID and division code</ExpectedInput>
		///		<ExpectedOutput>A data table with Antigen Typing test data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3769"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood unit Antigen Typing tests data
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit GUID</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns></returns>
		public static DataTable GetAntigenTypingTests(Guid bloodUnitGuid, string divisionCode)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetAntigenTypingTestsForUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetAntigenTypingTestsForUnit.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = divisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetAntigenTypingTestsForUnit.StoredProcName, prms).Tables[0];
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1661"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1662"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetAboConfirmationLookup
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAboConfirmationLookup()
		{
			return Common.StoredProcedure.GetData(STOREDPROC.GetAboConfirmationLookup.StoredProcName).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1663"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1664"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRHConfirmationLookup
		/// </summary>
		/// <returns></returns>
		public static DataTable GetRHConfirmationLookup()
		{
			return Common.StoredProcedure.GetData(STOREDPROC.GetRHConfirmationLookup.StoredProcName).Tables[0];
		}

	    ///<Developers>
	    ///	<Developer>Mike      </Developer>
	    ///</Developers>
	    ///<SiteName>Hines OIFO</SiteName>
	    ///<CreationDate>11/12/2002</CreationDate>
	    ///<TestCases>
	    ///	
	    ///<Case type="0" testid ="875"> 
	    ///		<ExpectedInput>DataTable containing test data, "V" Indicator, Limited Status</ExpectedInput>
	    ///		<ExpectedOutput>None</ExpectedOutput>
	    ///	</Case>
	    ///	
	    ///<Case type="0" testid ="7062"> 
	    ///		<ExpectedInput>DataTable containing test data, "W" Indicator, Limited Status</ExpectedInput>
	    ///		<ExpectedOutput>None</ExpectedOutput>
	    ///	</Case>
	    ///
	    ///<Case type="0" testid ="7267"> 
	    ///		<ExpectedInput>DataTable containing test data, "V" Indicator, non-Limited Status</ExpectedInput>
	    ///		<ExpectedOutput>None</ExpectedOutput>
	    ///	</Case>
	    ///
	    ///<Case type="0" testid ="7268"> 
	    ///		<ExpectedInput>DataTable containing test data, "W" Indicator, non-Limited Status</ExpectedInput>
	    ///		<ExpectedOutput>None</ExpectedOutput>
	    ///	</Case>
	    ///	
	    ///<Case type="0" testid ="8174"> 
	    ///		<ExpectedInput>Exception Non-Qc tested</ExpectedInput>
	    ///		<ExpectedOutput>None</ExpectedOutput>
	    ///	</Case>
	    ///
	    ///<Case type="1" testid ="2039"> 
	    ///		<ExpectedInput>Invalid DataTable</ExpectedInput>
	    ///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
	    ///	</Case>
	    ///</TestCases>
	    ///<Update></Update>
	    ///<ArchivePlan></ArchivePlan>
	    ///<Interfaces></Interfaces>
	    ///
	    ///<param name="bloodUnitTests">DataTable containing source blood unit test data</param>
	    ///<param name="dtExceptionUnitAboRH">DataTable containing exceptions</param>
	    /// <param name="testDateTime"></param>
	    /// <param name="testUser"></param>
	    /// <param name="automatedInstrName"></param>
	    /// <param name="automatedInstrQCd"></param>
	    /// <param name="rackGuid"></param>
	    ///<param name="divisionCode">Division code</param>
	    ///<param name="lastUpdateFunctionId">UC or method to call this function</param>
	    ///<param name="workloadEvents">Data table of workload event records</param>
	    ///<param name="dtExceptionNotQc">Exception to implement BR_56.07</param>
	    ///<summary>
	    /// This method iterates through the ABO/RH blood unit tests and updates the blood unit test table with all
	    /// valid test results.  If a record contains warnings the blood unit table quarantine indicator is set 
	    /// to true.  If is valid ('V') then the blood unit is marked as available.  
	    /// Implements BR_2.12
	    ///</summary>
        public static void SaveAboRHBloodUnitTests(DataTable bloodUnitTests, DataTable dtExceptionUnitAboRH, DataTable workloadEvents, DateTime testDateTime, string testUser, string automatedInstrName, bool automatedInstrQCd, Guid rackGuid, string divisionCode, Common.UpdateFunction lastUpdateFunctionId, DataTable dtExceptionNotQc)
        {
            string comment = string.Empty, unitLabelId = string.Empty;

            System.Guid worklistGuid;

            //We need 3 tables - 1 to hold all blood unit test result data,
            //1 to hold all blood unit status data and 1 to hold all the confirmation worklist unit data
            DataTable dtBloodUnitTests = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
            DataRow drBUT;

            DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
            DataRow drBUS = null;
            ArrayList alTables = new ArrayList();
            ArrayList alSprocs = new ArrayList();

            DataTable dtConfirmationWorklistUnit = GetEmptyConfirmationWorklistUnitTable();
            DataRow drCWU;

            // Get canned comment for quarantining unit with inconsistent ABO/Rh tests
            DataTable quarantineCommentTable = DAL.CannedComment.GetCannedComments(Common.LogonUser.LogonUserDivisionCode, "QD");
            Guid quarantineCommentGuid = (Guid)quarantineCommentTable.Rows[0][TABLE.CannedComment.CannedCommentGuid];

            foreach (DataRow _dr in bloodUnitTests.Rows)
            {
                //CR 2232, 2396
                drBUS = null;
                if (_dr[ARTIFICIAL.Indicator].ToString() == "V" || _dr[ARTIFICIAL.Indicator].ToString() == "W")
                {
                    worklistGuid = System.Guid.NewGuid();

                    //Next, create the necessary rows to insert into the 
                    //BloodUnitTest table with all Abo/Rh test results
                    unitLabelId = _dr[TABLE.BloodUnit.EyeReadableUnitId].ToString();
                    foreach (DataColumn _dc in _dr.Table.Columns)
                    {
                        int _testTypeId = int.MinValue;

                        if (_dr.IsNull(_dc) == false)
                        {
                            if (_dr[_dc.Ordinal].ToString() != string.Empty)
                            {
                                drBUT = dtBloodUnitTests.NewRow();
                                drBUT[TABLE.BloodUnitTest.BloodUnitTestGuid] = System.Guid.NewGuid();
                                comment = string.Empty;

                                if (_dc.ColumnName == TABLE.ValidRhResult.RhInterp)
                                {
                                    _testTypeId = (int)Common.TestType.RhInterp;
                                    comment = _dr[ARTIFICIAL.RhComment].ToString();

                                }
                                if (_dc.ColumnName == TABLE.ValidAboResult.AboInterp)
                                {
                                    _testTypeId = (int)Common.TestType.ABOInterp;
                                    comment = _dr[ARTIFICIAL.AboComment].ToString();

                                }


                                if (_dc.ColumnName == TABLE.ValidAboResult.AntiA) _testTypeId = (int)Common.TestType.AntiA;
                                if (_dc.ColumnName == TABLE.ValidAboResult.AntiB) _testTypeId = (int)Common.TestType.AntiB;
                                if (_dc.ColumnName == TABLE.ValidAboResult.AntiAB) _testTypeId = (int)Common.TestType.AntiAB;
                                if (_dc.ColumnName == TABLE.ValidRhResult.AntiD) _testTypeId = (int)Common.TestType.AntiD;
                                if (_dc.ColumnName == TABLE.ValidRhResult.DControl) _testTypeId = (int)Common.TestType.ControlD;

                                if (_testTypeId != int.MinValue)
                                {
                                    //Create a Test Record -- Once for each testworthy column
                                    drBUT[TABLE.BloodUnitTest.BloodUnitGuid] = _dr[TABLE.BloodUnitTest.BloodUnitGuid];
                                    drBUT[TABLE.BloodUnitTest.BloodTestTypeId] = _testTypeId;
                                    drBUT[TABLE.BloodUnitTest.TestDate] = testDateTime;
                                    drBUT[TABLE.BloodUnitTest.TestTechId] = testUser;
                                    drBUT[TABLE.BloodUnitTest.TestResultId] = _dr[_dc.ColumnName].ToString();
                                    drBUT[TABLE.BloodUnitTest.EntryMethodCode] = Convert.ToString(Common.TestEntryMethod.Manual);
                                    drBUT[TABLE.BloodUnitTest.TestComments] = comment;
                                    drBUT[TABLE.BloodUnitTest.DivisionCode] = divisionCode;
                                    drBUT[TABLE.BloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                                    drBUT[TABLE.BloodUnitTest.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
                                    drBUT[TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid] = worklistGuid;

                                    drBUT[TABLE.BloodUnitTest.EntryTechId] = testUser;
                                    drBUT[TABLE.BloodUnitTest.TestingMethodCode] = (automatedInstrName == string.Empty) ? "U" : "A";
                                    drBUT[TABLE.BloodUnitTest.RackGuid] = rackGuid;
                                    drBUT[TABLE.BloodUnitTest.AutoInstrumentName] = automatedInstrName;

                                    dtBloodUnitTests.Rows.Add(drBUT);
                                }
                            }
                        }
                    }

                    //Here, insert the ConfirmationWorklistUnit data
                    drCWU = dtConfirmationWorklistUnit.NewRow();
                    drCWU[TABLE.ConfirmationWorklistUnit.BloodUnitGuid] = new Guid(_dr[TABLE.BloodUnitTest.BloodUnitGuid].ToString());
                    drCWU[TABLE.ConfirmationWorklistUnit.ConfirmationWorklistUnitGuid] = worklistGuid;
                    drCWU[TABLE.ConfirmationWorklistUnit.RackGuid] = rackGuid;
                    drCWU[TABLE.ConfirmationWorklistUnit.AutoInstrumentName] = automatedInstrName;
                    drCWU[TABLE.ConfirmationWorklistUnit.AutoInstrumentQcIndicator] = automatedInstrQCd;
                    drCWU[TABLE.ConfirmationWorklistUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drCWU[TABLE.ConfirmationWorklistUnit.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
                    drCWU[TABLE.ConfirmationWorklistUnit.SortColumn] = 0;
                    dtConfirmationWorklistUnit.Rows.Add(drCWU);

                   System.Guid bloodUnitGuid = (System.Guid)_dr[TABLE.BloodUnitTest.BloodUnitGuid];
                   DataTable dtExistingBUSData = DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(bloodUnitGuid);

                   drBUS = dtBloodUnitStatus.NewRow();
                   foreach (System.Data.DataColumn dc in dtBloodUnitStatus.Columns)
                    {
                        if (dtExistingBUSData.Columns.Contains(dc.ColumnName))
                        {
                            if (dc.ColumnName == TABLE.BloodUnitStatus.RowVersion)
                            {
                                //Use the original rowversion, so we're not cheating and potentially updating modified data since our tests began
                                drBUS[dc.ColumnName] = _dr[TABLE.BloodUnitStatus.RowVersion];
                            }
                            else
                            {
                                drBUS[dc.ColumnName] = dtExistingBUSData.Rows[0][dc.ColumnName];
                            }
                        }
                    }
                   //Now, change the blood unit status as appropriate
                    if (_dr[ARTIFICIAL.Indicator].ToString() == "V")
                    {
                        //TestingLimitIndicator flag will trip the fnUnitStatusCode to keep the unit status 
                        drBUS[TABLE.BloodUnitStatus.TestingLimitIndicator] = false;
                    }
                    else
                    {
                        drBUS[TABLE.BloodUnitStatus.TestingLimitIndicator] = true;
                        drBUS[TABLE.BloodUnitStatus.QuarantineIndicator] = 1;
                        drBUS[TABLE.BloodUnitStatus.QuarantineCannedCommentGuid] = quarantineCommentGuid;
                    }
                    dtBloodUnitStatus.Rows.Add(drBUS);
                }
            }

            alTables.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitStatus, lastUpdateFunctionId));
            alTables.Add(dtConfirmationWorklistUnit);
            alTables.Add(dtBloodUnitTests);
            alTables.Add(Common.Utility.AppendLastUpdateInformation(workloadEvents, lastUpdateFunctionId));

            alSprocs.Add(STOREDPROC.UpdateBloodUnitStatus.StoredProcName);
            alSprocs.Add(STOREDPROC.InsertConfirmationWorklistUnit.StoredProcName);
            alSprocs.Add(STOREDPROC.InsertBloodUnitTest.StoredProcName);
            alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);



            if (dtExceptionUnitAboRH != null)
            {
                alTables.Add(dtExceptionUnitAboRH);
                alSprocs.Add(STOREDPROC.InsertExceptionUnitAboRH.StoredProcName);
            }

            if (dtExceptionNotQc != null)		//BR_56.07
            {
                alTables.Add(dtExceptionNotQc);
                alSprocs.Add(STOREDPROC.InsertExceptionQCNotPerformed.StoredProcName);
            }

            int retVal = (new Common.StoredProcedure()).TransactionalGetValue(alSprocs, alTables);
        }

	    ///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7514"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7515"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetEmptyBloodUnitTestTable
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static DataTable GetEmptyBloodUnitTestTableSchema(bool isUpdate)
		{
			DataTable dtEmpty = new DataTable(TABLE.BloodUnitTest.TableName);

			dtEmpty.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.BloodUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.BloodTestTypeId, typeof(int));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestResultId, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.EntryMethodCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid, typeof(System.Guid));

			//Adding 4 new fields that seem like they should be populated, since we have the data
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.EntryTechId, typeof(string));			//2-2-Lohse
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestingMethodCode, typeof(string));	//2-2-Lohse	
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.RackGuid, typeof(System.Guid));		//2-2-Lohse
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.AutoInstrumentName, typeof(string));	//2-2-Lohse

			//01/17/2006 Lohse - Adding RecordStatusCode so it can be set by the modification UCs
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.RecordStatusCode, typeof(string));
			if (isUpdate)
			{
				dtEmpty.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			}
			
			return(dtEmpty);
		}

		/// <summary>
		/// GetEmptyConfirmationWorklistUnitTable
		/// </summary>
		/// <returns></returns>
		private static DataTable GetEmptyConfirmationWorklistUnitTable()
		{
			DataTable dtEmpty = new DataTable(TABLE.ConfirmationWorklistUnit.TableName);

			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.BloodUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.ConfirmationWorklistUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.RackGuid, typeof(Guid));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.AutoInstrumentName);
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.AutoInstrumentQcIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.ConfirmationWorklistUnit.SortColumn, typeof(int));

			return(dtEmpty);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="643"> 
		///		<ExpectedInput>Blood unit test data</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1898"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves new blood unit test
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit GUID</param>
		/// <param name="bloodTestTypeId">Blood test type ID</param>
		/// <param name="testResultId">Test result ID</param>
		/// <param name="entryMethodCode">Entry method code</param>
		/// <param name="testComments">Test comments</param>
		/// <param name="divisionCode">Division code</param>
		/// <param name="lastUpdateFunctionId">UC or calling Method</param>
		/// <param name="confirmationWorklistUnitGuid">Worklist Guid</param>
		public static void SaveBloodUnitTest(Guid bloodUnitGuid, int bloodTestTypeId,
		                                     string testResultId, string entryMethodCode,
		                                     string testComments, string divisionCode, Common.UpdateFunction lastUpdateFunctionId, System.Guid confirmationWorklistUnitGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.bloodtesttypeid, System.Data.SqlDbType.Int),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.testdate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.testresultid, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.entrymethodcode, System.Data.SqlDbType.Char),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.testcomments, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.divisioncode, System.Data.SqlDbType.Char),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.lastupdateuser, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.lastupdatefunctionid, System.Data.SqlDbType.Int),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.confirmationworklistunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.InsertBloodUnitTest.bloodunittestguid, System.Data.SqlDbType.UniqueIdentifier)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = bloodTestTypeId;
			prms[2].Value = VbecsDateTime.GetDivisionCurrentDateTime();
			prms[3].Value = testResultId;
			prms[4].Value = entryMethodCode;
			prms[5].Value = testComments;
			prms[6].Value = divisionCode;
			prms[7].Value = Common.LogonUser.LogonUserName;
			prms[8].Value = (int) lastUpdateFunctionId;
			prms[9].Value = confirmationWorklistUnitGuid;
			prms[10].Value = System.Guid.NewGuid();

			if (confirmationWorklistUnitGuid == System.Guid.Empty)
			{
				prms[9].Value = DBNull.Value;
			}

			int returnValue = ((new Common.StoredProcedure()).TransactionalGetValue(STOREDPROC.InsertBloodUnitTest.StoredProcName, prms));
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/2/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6781"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Datatable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6782"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Implements BR_2.03
		/// </summary>
		/// <returns></returns>
		public static DataTable GetEmptyAboRhConfirmationTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable();		//Can't have a TABLE Name otherwise the DataGrid goes LOCO!

			dtEmpty.Columns.Add(ARTIFICIAL.Indicator, typeof(String));
			dtEmpty.Columns.Add(TABLE.BloodUnit.EyeReadableUnitId, typeof (String));
			dtEmpty.Columns.Add(TABLE.ValidAboResult.AntiA, typeof(String));
			dtEmpty.Columns.Add(TABLE.ValidAboResult.AntiB, typeof(String));
			dtEmpty.Columns.Add(TABLE.ValidAboResult.AntiAB, typeof(String));
			dtEmpty.Columns.Add(TABLE.ValidRhResult.AntiD, typeof(String));
			dtEmpty.Columns.Add(TABLE.ValidRhResult.DControl, typeof(String));			
			dtEmpty.Columns.Add(TABLE.ValidAboResult.AboInterp, typeof(String));			
			dtEmpty.Columns.Add(TABLE.ValidRhResult.RhInterp, typeof(String));			
			dtEmpty.Columns.Add(ARTIFICIAL.AboComment, typeof(String));
			dtEmpty.Columns.Add(ARTIFICIAL.RhComment, typeof(String));
			dtEmpty.Columns.Add(TABLE.BloodUnit.BloodUnitGuid, typeof(Guid));
			dtEmpty.Columns.Add(TABLE.ValidAboResult.AboInitial, typeof(String));
			dtEmpty.Columns.Add(TABLE.ValidRhResult.RhInitial, typeof(Boolean));
			dtEmpty.Columns.Add(ARTIFICIAL.IndicatorMessage, typeof(String));
			dtEmpty.Columns.Add(TABLE.BloodUnitStatus.RowVersion, typeof(Byte[]));
			dtEmpty.Columns.Add(TABLE.BloodUnitStatus.BloodUnitStatusGuid,typeof(Guid));

			dtEmpty.Columns.Add(ARTIFICIAL.BR217, typeof(Object));		//Mismatch ABO/RH (Exception)
			dtEmpty.Columns.Add(ARTIFICIAL.BR221, typeof(Object));		//Interp = Inconclusive (Exception)
			dtEmpty.Columns.Add(ARTIFICIAL.BR207 , typeof(bool));			//Rh is required
			dtEmpty.Columns.Add(TABLE.BloodUnitStatus.QuarantineIndicator, typeof(bool));	//CR692
			dtEmpty.Columns.Add(TABLE.BloodProduct.ProductCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode, typeof(string));
			
			return(dtEmpty);
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="644"> 
		///		<ExpectedInput>Blood unit test identifier and test result identifier</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1899"> 
		///		<ExpectedInput>Bad RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates test result of an existing blood unit test with a given GUID.
		/// </summary>
		/// <param name="bloodUnitTestGuid">Blood unit test GUID</param>
		/// <param name="testResultId">Test result ID</param>
		/// <param name="testDate">Test Date</param>
		/// <param name="testTechId">Test Tech ID</param>
		/// <param name="rowVersion">Row Version</param>
		/// <param name="lastUpdateFunctionId">UC or calling Method</param>
		public static void UpdateTestResult(Guid bloodUnitTestGuid, string testResultId, DateTime testDate, string testTechId, byte[] rowVersion, Common.UpdateFunction lastUpdateFunctionId)
		{
			SqlParameter[] prms = {
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.bloodunittestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.testresultid, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.testdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.testtechid, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.entrytechid, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.lastupdateuser, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.lastupdatefunctionid, System.Data.SqlDbType.Int),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.rowversion, System.Data.SqlDbType.Binary),
				new SqlParameter(STOREDPROC.UpdateBloodUnitTestResult.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = bloodUnitTestGuid;
			prms[1].Value = testResultId;
			prms[2].Value = testDate;
			prms[3].Value = testTechId;
			prms[4].Value = Common.LogonUser.LogonUserName;
			prms[5].Value = Common.LogonUser.LogonUserName;
			prms[6].Value = (int) lastUpdateFunctionId;
			prms[7].Value = rowVersion;
			prms[8].Value = Common.LogonUser.LogonUserDivisionCode;

			int returnValue = ((new Common.StoredProcedure()).TransactionalGetValue(STOREDPROC.UpdateBloodUnitTestResult.StoredProcName, prms));
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1667"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1668"> 
		///		<ExpectedInput>Datatables with bad RowVersions</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SaveAntigenTypingBloodUnitTests
		/// </summary>
		/// <param name="bloodUnitTests">bloodUnitTests</param>
		/// <param name="rackTests">rackTests</param>
		/// <param name="qcNewRackLots">qcNewRackLots</param>
		/// <param name="qcRackLots">qcRackLots</param>
		/// <param name="qcAntiseraUnitTests">qcAntiseraUnitTests</param>
		/// <param name="dtExceptionExpiredReagentQC">dtExceptionExpiredReagentQC</param>
		/// <param name="dtExceptionUnitIssueTest">dtExceptionUnitIssueTest</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="workloadEvents">Workload Event data</param>
		/// <returns>True or False</returns>
		public static bool SaveAntigenTypingBloodUnitTests(DataTable workloadEvents, DataTable bloodUnitTests, DataTable rackTests, DataTable qcNewRackLots, DataTable qcRackLots, DataTable qcAntiseraUnitTests, DataTable dtExceptionExpiredReagentQC, DataTable dtExceptionUnitIssueTest, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();

			dtArray.Add(Common.Utility.AppendLastUpdateInformation(bloodUnitTests, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.UpdateBloodUnitTestResult.StoredProcName);
			
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(qcNewRackLots, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.InsertRackLot.StoredProcName);
			
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(qcRackLots, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.UpdateRackLot.StoredProcName);
			
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(rackTests, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.UpdateRackTest.StoredProcName);
			
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(qcAntiseraUnitTests, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.InsertAntiseraUnitTest.StoredProcName);
			
			if (dtExceptionExpiredReagentQC != null)
			{
				dtArray.Add(dtExceptionExpiredReagentQC);
				sprocArray.Add(STOREDPROC.InsertExceptionQC.StoredProcName);
			}

			if (dtExceptionUnitIssueTest != null)
			{
				dtArray.Add(dtExceptionUnitIssueTest);
				sprocArray.Add(STOREDPROC.InsertExceptionEmergencyIssueUnitTest.StoredProcName);
			}

			if(workloadEvents != null)
			{
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(workloadEvents, lastUpdateFunctionId));
				sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}
	
			int retVal = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			return (retVal == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1669"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1670"> 
		///		<ExpectedInput>DataTables with bad RowVersions</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SaveWeakDBloodUnitTests
		/// </summary>
		/// <param name="bloodUnitTests">Blood Unit Tests</param>
		/// <param name="dtExceptionUnitIssueTest">Unit Issue test mismatch exception</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="workloadEvents">Workload event data</param>
		/// <returns>True or False</returns>
		public static bool SaveWeakDBloodUnitTests(DataTable bloodUnitTests, DataTable workloadEvents, DataTable dtExceptionUnitIssueTest, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();

			dtArray.Add(Common.Utility.AppendLastUpdateInformation(workloadEvents, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);

			dtArray.Add(Common.Utility.AppendLastUpdateInformation(bloodUnitTests, lastUpdateFunctionId));
			sprocArray.Add(STOREDPROC.UpdateBloodUnitTestResult.StoredProcName);

			if (dtExceptionUnitIssueTest != null)
			{
				dtArray.Add(dtExceptionUnitIssueTest);
				sprocArray.Add(STOREDPROC.InsertExceptionEmergencyIssueUnitTest.StoredProcName);
			}

			int retVal = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			return(retVal == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1671"> 
		///		<ExpectedInput>OrderedTestGuid, OrderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1672"> 
		///		<ExpectedInput>Invalid Guids</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTestResultsForXM
		/// </summary>
		/// <param name="orderedTestGuid">OrderedTestGuid</param>
		/// <param name="orderedComponentGuid">OrderedComponentGuid</param>
		/// <returns>Data table</returns>
		public static DataTable GetTestResultsForXM(Guid orderedTestGuid, Guid orderedComponentGuid)
		{
			SqlParameter[] prms = new SqlParameter[1];

			if (orderedTestGuid == Guid.Empty)
			{
				prms[0] = new SqlParameter(STOREDPROC.GetTestResultsForXM.OrderedComponentGuid, System.Data.SqlDbType.UniqueIdentifier);
				prms[0].Value = orderedComponentGuid;
				return Common.StoredProcedure.GetData(STOREDPROC.GetTestResultsForXM.StoredProcName, prms).Tables[0];
			}
			else
			{
				prms[0] = new SqlParameter(STOREDPROC.GetTestResultsForRepeatXM.OrderedTestGuid, System.Data.SqlDbType.UniqueIdentifier);
				prms[0].Value = orderedTestGuid;
				return Common.StoredProcedure.GetData(STOREDPROC.GetTestResultsForRepeatXM.StoredProcName, prms).Tables[0];
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3770"> 
		///		<ExpectedInput>Valid blood unit guid, confirmation worklist unit guid, comment and last update function id</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3771"> 
		///		<ExpectedInput>Invalid blood unit guid, confirmation worklist unit guid, comment and last update function id</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update>
		///12/6/2004 RDH
		///		Added dtABORHConfirmationTests to parameter list.
		///		The dtABORHConfirmationTests datatable is now built and
		///		populated in the business object layer.
		///	05/27/2005 DWA
		///		Removed superfluous parameters
		///</Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivate a set of ABO/Rh confirmation test results
		/// </summary>
		/// <param name="dtABORHConfirmationTests">tests to invalidate</param>
		/// <param name="dtWorkload">workload data for this process</param>
		public static int InactivateAboRHConfirmationTests( DataTable dtABORHConfirmationTests, DataTable dtWorkload )
		{
			System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList spArray = new ArrayList();
			//
			dtArray.Add(dtABORHConfirmationTests);
			dtArray.Add(dtWorkload);
			//
			spArray.Add(STOREDPROC.InactivateAboRHConfirmationTests.StoredProcName);
			spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			//
			return new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray);
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3772"> 
		///		<ExpectedInput>Valid blood unit guid, antisera worklist typing guid, orderable test id, comment and last update function id</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3773"> 
		///		<ExpectedInput>Invalid blood unit guid, antisera worklist typing guid, orderable test id, comment and last update function id</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivate a set of antigen typing test results
		/// </summary>
		/// <param name="dtAntigenTypingTests">Tests to invalidate</param>
		/// <param name="dtWorkloadEvent">Workload event data</param>
		/// <param name="lastUpdateFunctionId">UC or function performing this call</param>
		public static bool InactivateAntigenTypingTests
		(
			DataTable dtAntigenTypingTests,
			DataTable dtWorkloadEvent,
			Common.UpdateFunction lastUpdateFunctionId
		)
		{
			System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList spArray = new ArrayList();

			dtArray.Add(dtAntigenTypingTests);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, lastUpdateFunctionId));
	
			spArray.Add(STOREDPROC.InactivateAntigenTypingTests.StoredProcName);
			spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);

			return (new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4487"> 
		///		<ExpectedInput>Valid DataTables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4488"> 
		///		<ExpectedInput>Invalid BloodUnitTestResults table</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function invalidates specimen test results, and marks the specimen as unacceptable (UC_94)
		/// </summary>
		/// <param name="bloodUnitTestResults">Blood unit test table</param>
		/// <param name="orderedUnit">Ordered unit table</param>
		/// <param name="orderedTest">Ordered test table (optional)</param>
		/// <param name="orderedComponent">Ordered component table (optional)</param>
		/// <param name="lastUpdateFunctionId">Last update function ID</param>
		/// <param name="dtWorkloadEvent">workload events</param>
		/// <returns>Success indicator</returns>
		public static bool InvalidateCrossmatchTestResults(DataTable bloodUnitTestResults, DataTable orderedUnit, DataTable orderedTest, DataTable orderedComponent, DataTable dtWorkloadEvent, Common.UpdateFunction lastUpdateFunctionId)
		{
			Common.Utility.RequireNonNullColumns(bloodUnitTestResults, STOREDPROC.UpdateCrossmatchTestStatus.RequiredParameters);
			//
			if ((orderedUnit != null) && (orderedUnit.Rows.Count > 0))
			{
				Common.Utility.RequireNonNullColumns(orderedUnit, STOREDPROC.UpdateOrderedUnit.RequiredParameters);
			}
			//
			if ((orderedTest != null) && (orderedTest.Rows.Count > 0))
			{
				Common.Utility.RequireNonNullColumns(orderedTest, STOREDPROC.UpdateOrderedTestStatus.RequiredParameters);
			}
			//
			if ((orderedComponent != null) && (orderedComponent.Rows.Count > 0))
			{
				Common.Utility.RequireNonNullColumns(orderedComponent, STOREDPROC.UpdateOrderedComponentStatus.RequiredParameters);
			}
			//
			ArrayList dtArray = new ArrayList();
			ArrayList spArray = new ArrayList();
			//
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(bloodUnitTestResults, lastUpdateFunctionId));
			spArray.Add(STOREDPROC.UpdateCrossmatchTestStatus.StoredProcName);
			//
			if ((orderedUnit != null) && (orderedUnit.Rows.Count > 0))
			{
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedUnit, lastUpdateFunctionId));
				spArray.Add(STOREDPROC.UpdateOrderedUnit.StoredProcName);
			}
			//
			if ((orderedTest != null) && (orderedTest.Rows.Count > 0))
			{
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedTest, lastUpdateFunctionId));
				spArray.Add(STOREDPROC.UpdateOrderedTestStatus.StoredProcName);
			}
			//
			if ((orderedComponent != null) && (orderedComponent.Rows.Count > 0))
			{
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedComponent, lastUpdateFunctionId));
				spArray.Add(STOREDPROC.UpdateOrderedComponentStatus.StoredProcName);
			}
			//
			if(dtWorkloadEvent != null)
			{
				if(dtWorkloadEvent.Rows.Count > 0)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, lastUpdateFunctionId));
					spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				}
			}
			int result = new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray);

			return (result == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1676"> 
		///		<ExpectedInput>BloodUnitTestGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1677"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to retrieve the parent ordered test or ordered component
		/// of a given blood unit test entry
		/// </summary>
		/// <param name="bloodUnitTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitTestOrder(Guid bloodUnitTestGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitTestOrder.bloodunittestguid, System.Data.SqlDbType.UniqueIdentifier),
				};

			prms[0].Value = bloodUnitTestGuid;

			return Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestOrder.StoredProcName, prms).Tables[0];
		}



		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7524"> 
		///		<ExpectedInput>Valid DataTable</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7525"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivate partially completed crossmatch results
		/// </summary>
		/// <param name="dtInvalidBloodUnitTest"></param>
		/// <returns></returns>
		public static bool InvalidatePartiallyCompletedCrossmatch(DataTable dtInvalidBloodUnitTest)
		{
			int result = new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.InvalidateBloodUnitTestsForReleasedOrderedUnit.StoredProcName, dtInvalidBloodUnitTest);

			return (result == 0);
		}
	}
}